#!/bin/bash
# Copyright (c) 2025 by Ivanti, Inc. All rights reserved

# Security hardening: Unset standard commands to prevent function override attacks
unset -f printf echo grep sed awk wget dpkg apt fuser timeout mkdir chmod rm mv cp
unset -f readlink realpath command test


if [ -t 1 ] && [ -n "$TERM" ] && tput setaf 1 &>/dev/null; then
    RED='\033[0;31m'; GREEN='\033[0;32m'; NC='\033[0m'
else
    RED=''; GREEN=''; NC=''
fi
RED_X="${RED}✗${NC}"; GREEN_CHECK="${GREEN}✓${NC}"; RED_MISSING="${RED}(missing)${NC}"; GREEN_SUCCESS="${GREEN}(successfully installed)${NC}"

# Log file and temp directory paths
LOG_FILE="/var/log/pulsesecure/isac_dependency_installer.log"
readonly LOG_FILE
TEMP_DIR="/tmp/isac_deps"
readonly TEMP_DIR
DPKG_CMD="$(command -v dpkg || echo /usr/bin/dpkg)"
readonly DPKG_CMD
WGET_CMD="$(command -v wget || echo /usr/bin/wget)"
readonly WGET_CMD
FILE_CMD="$(command -v file || echo /usr/bin/file)"
readonly FILE_CMD
SUDO_CMD="$(command -v sudo || echo /usr/bin/sudo)"
readonly SUDO_CMD
APT_CMD="$(command -v apt || echo /usr/bin/apt)"
readonly APT_CMD


# Global associative array mapping package names to their download paths in Ubuntu repositories
# Structure: [package_name]=relative_path_from_pool/main  
# Used to locate and download specific package versions for Ubuntu 24.04 compatibility
declare -gA PACKAGE_PATHS=(
    ["libwebkit2gtk-4.0-37"]="w/webkit2gtk/libwebkit2gtk-4.0-37_2.20.1-1_amd64.deb"
    ["libicu60"]="i/icu/libicu60_60.2-3ubuntu3_amd64.deb"
    ["libjavascriptcoregtk-4.0-18"]="w/webkit2gtk/libjavascriptcoregtk-4.0-18_2.20.1-1_amd64.deb"
    ["libwebp6"]="libw/libwebp/libwebp6_0.6.1-2_amd64.deb"
    ["multiarch-support"]="g/glibc/multiarch-support_2.27-3ubuntu1.6_amd64.deb"
    ["libhunspell-1.3-0"]="h/hunspell/libhunspell-1.3-0_1.3.3-4ubuntu1_amd64.deb"
    ["libenchant1c2a"]="e/enchant/libenchant1c2a_1.6.0-10ubuntu1_amd64.deb"
)

# Global associative array defining package dependency relationships
# Structure: [main_package]=comma_separated_list_of_dependencies
# Dependencies are installed before the main package to ensure proper installation order
declare -gA PACKAGE_DEPENDENCIES=(
    ["libwebkit2gtk-4.0-37"]="multiarch-support,libicu60,libwebp6,libhunspell-1.3-0,libenchant1c2a,libjavascriptcoregtk-4.0-18"
)

# Array of trusted repository domains for security validation
readonly TRUSTED_REPOS=("archive.ubuntu.com")

# Exit immediately on error, unset variable, or failed pipeline (robust error handling)
set -euo pipefail

# Writes a message to the installer log file.
log() { printf '%s\n' "$*" >> "$LOG_FILE" 2>/dev/null; }
log_and_console() { log "$*"; [ "${SILENT:-false}" = "true" ] || printf '%s\n' "$*"; }
error_exit() { log "ERROR: $*"; [ "${SILENT:-false}" != "true" ] && printf 'ERROR: %s\n' "$*" >&2; exit 1; }
is_installed() { dpkg-query -W -f='${Status}' "$1" 2>/dev/null | grep -q "install ok installed"; }

init_env() {
    if [ "$EUID" -ne 0 ]; then
        # Check if readlink is available
        if ! command -v readlink >/dev/null 2>&1; then
            echo "readlink is not available. Please install coreutils."
            exit 1
        fi
        # Resolve the absolute path of the script
        SCRIPT_PATH="$(readlink -f "$0")"
        if [ ! -f "$SCRIPT_PATH" ]; then
            echo "Failed to resolve script path for privilege escalation."
            exit 1
        fi
        # Re-execute the script with sudo, preserving all arguments
        exec "$SUDO_CMD" "$SCRIPT_PATH" "$@"
    fi
    # Platform configuration for Ubuntu 24.04 (must be inside function for sudo re-exec)
    declare -gA UBUNTU_24_04=(
        [repo]='https://archive.ubuntu.com/ubuntu/pool/main'
        [pkgs]='libwebkit2gtk-4.0-37'
    )
    mkdir -p "${LOG_FILE%/*}" || error_exit "Cannot create log directory: ${LOG_FILE%/*}"
    > "$LOG_FILE" 2>/dev/null || error_exit "Cannot create log file: $LOG_FILE"
    chmod 644 "$LOG_FILE" 2>/dev/null || error_exit "Cannot set permissions on log file: $LOG_FILE"
}

cleanup() { test -d "$TEMP_DIR" && rm -rf "$TEMP_DIR"; }
trap cleanup EXIT

detect_platform() {
    test -f /etc/os-release || error_exit "OS detection failed - missing /etc/os-release"
    . /etc/os-release || error_exit "Cannot read OS information"
    
    local platform_key="${ID:-unknown}:${VERSION_ID:-unknown}"
    case "$platform_key" in
        ubuntu:24.04) printf 'UBUNTU_24_04' ;;
        # Future platform support - commented out as per PR review:
        # ubuntu:22.04) printf 'UBUNTU_22_04' ;;
        # debian:12) printf 'DEBIAN_12' ;;
        *) 
            log "Detected platform: $platform_key"
            log "Currently supported: Ubuntu 24.04 only"
            error_exit "Unsupported platform: $platform_key"
        ;;
    esac
}

install_package() {
    local pkg="$1" repo="$2" path="${3:-}" 
    
    # Check if already installed first
    if is_installed "$pkg"; then
        log "✓ $pkg (already installed)"; return 0
    fi

    # Only manual download and install using apt (not dpkg) for dependency resolution
    test -n "$path" || { log "✗ $pkg (no download path provided)"; return 1; }
    local url="$repo/$path" deb_file="$TEMP_DIR/${path##*/}"
    log "[START] $pkg"
    log "URL: $url"
    
    # Extract domain from URL and validate against trusted repositories
    url_domain=$(printf '%s' "$url" | sed -E 's|^https://([^/]+)/.*$|\1|')
    is_trusted=0
    for trusted in "${TRUSTED_REPOS[@]}"; do
        if [ "$url_domain" = "$trusted" ]; then
            is_trusted=1
            break
        fi
    done
    if [ "$is_trusted" -ne 1 ]; then
        log "✗ $pkg (untrusted repository domain: $url_domain)"
        return 1
    fi
    # Validate URL format (https, trusted domain, .deb extension)
    printf '%s' "$url" | grep -Eq '^https://[^/]+/.*\.deb$' || { log "✗ $pkg (invalid URL format: $url)"; return 1; }
    mkdir -p "$TEMP_DIR" || { log "✗ $pkg (temp creation failed)"; return 1; }
    log "Downloading $pkg from $url to $deb_file"
    # Download the package
    if timeout 300 "$WGET_CMD" --https-only --secure-protocol=auto -q "$url" -O "$deb_file" 2>>"$LOG_FILE"; then
        log "Downloaded: $pkg ($deb_file)"
    else
        log "✗ $pkg (download failed from $url)"
        return 1
    fi
    # Check for apt/dpkg locks before installing
    for lock in /var/lib/dpkg/lock /var/lib/dpkg/lock-frontend /var/lib/apt/lists/lock /var/cache/apt/archives/lock; do
        if fuser "$lock" >/dev/null 2>&1; then
            log "✗ $pkg (installation blocked: another package operation in progress)"
            return 1
        fi
    done
    log "Installing $pkg using apt from $deb_file"
    if "$APT_CMD" install -y "$deb_file" >>"$LOG_FILE" 2>&1; then
        log "✓ $pkg (successfully installed from $deb_file)"
        return 0
    else
        log "✗ $pkg (apt installation failed for $deb_file)"
        return 1
    fi
}


process_packages() {
    local platform="$1" check_only="$2"
    local repo="" packages=""
    if declare -p UBUNTU_24_04 &>/dev/null && [ "$platform" = "UBUNTU_24_04" ]; then
        repo="${UBUNTU_24_04[repo]:-}"
        packages="${UBUNTU_24_04[pkgs]:-}"
    fi
    if [ -z "$repo" ] || [ -z "$packages" ]; then
        error_exit "Invalid platform configuration (repo or pkgs missing)"
    fi

    local installed=0 missing=0 failed=0
    IFS=' ' read -ra pkg_list <<< "$packages"

        if [ ${#pkg_list[@]} -gt 0 ]; then
            log "$([ "$check_only" = "true" ] && echo "Check-only mode:" || echo "Install mode:") Checking ${#pkg_list[@]} dependencies..."
            log_and_console "Checking dependencies..."
    fi

    for pkg in "${pkg_list[@]}"; do
        # Install dependencies first (if any)
        if [[ -v PACKAGE_DEPENDENCIES[$pkg] ]] && [[ -n "${PACKAGE_DEPENDENCIES[$pkg]}" ]]; then
            IFS=',' read -ra dep_list <<< "${PACKAGE_DEPENDENCIES[$pkg]}"
            for dep in "${dep_list[@]}"; do
                if is_installed "$dep"; then
                    log "Details: $dep is present. Version: $($DPKG_CMD -s $dep 2>/dev/null | grep Version)"
                else
                    if [ "$check_only" = "true" ]; then
                        # Print 'missing' in red
                        log "✗ $dep (missing)"
                        if [ "${SILENT:-false}" != "true" ]; then
                            printf '%b✗%b %s %b(missing)%b\n' "$RED" "$NC" "$dep" "$RED" "$NC"
                        fi
                        ((missing++))
                    else
                        # Print only '(missing)' in red, '(installing...)' normal
                        log "✗ $dep (missing - installing...)"
                        if [ "${SILENT:-false}" != "true" ]; then
                            printf '%b✗%b %s %b(missing)%b - installing...\n' "$RED" "$NC" "$dep" "$RED" "$NC"
                        fi
                        local dep_path=""
                        if [[ -v PACKAGE_PATHS[$dep] ]]; then
                            dep_path="${PACKAGE_PATHS[$dep]}"
                        fi
                        if [ -n "$dep_path" ]; then
                            if install_package "$dep" "$repo" "$dep_path"; then
                                if [ "${SILENT:-false}" != "true" ]; then
                                    printf '%b✓%b %s %b(dependency installed successfully)%b\n' "$GREEN" "$NC" "$dep" "$GREEN" "$NC"
                                fi
                            else
                                log_and_console "✗ $dep (dependency installation failed)"
                                ((failed++))
                            fi
                        else
                            log_and_console "✗ $dep (no download path, skipped)"
                            ((failed++))
                        fi
                    fi
                fi
            done
        fi
        
        # Now install the main package (only if dependencies succeeded or check-only mode)
        local pkg_path=""
        if [[ -v PACKAGE_PATHS[$pkg] ]]; then
            pkg_path="${PACKAGE_PATHS[$pkg]}"
        fi
        if is_installed "$pkg"; then
            ((installed++))
            log "✓ $pkg (already installed)"
            log "Details: $pkg is present. Version: $($DPKG_CMD -s $pkg 2>/dev/null | grep Version)"
        else
            if [ "$check_only" = "true" ]; then
                # Print 'missing' in red
                log "✗ $pkg (missing)"
                if [ "${SILENT:-false}" != "true" ]; then
                    printf '%b✗%b %s %b(missing)%b\n' "$RED" "$NC" "$pkg" "$RED" "$NC"
                fi
                ((missing++))
            else
                # Print only '(missing)' in red, '(installing...)' normal
                log "✗ $pkg (missing - installing...)"
                if [ "${SILENT:-false}" != "true" ]; then
                    printf '%b✗%b %s %b(missing)%b - installing...\n' "$RED" "$NC" "$pkg" "$RED" "$NC"
                fi
                if [ -n "$pkg_path" ]; then
                    if install_package "$pkg" "$repo" "$pkg_path"; then
                        ((installed++))
                        # Only print success once
                        if [ "${SILENT:-false}" != "true" ]; then
                            printf '%b✓%b %s %b(successfully installed)%b\n' "$GREEN" "$NC" "$pkg" "$GREEN" "$NC"
                        fi
                        log "✓ $pkg (successfully installed)"
                    else
                        log_and_console "✗ $pkg (installation failed)"
                        ((failed++))
                    fi
                else
                    log_and_console "✗ $pkg (no download path, skipped)"
                    ((failed++))
                fi
            fi
        fi
    done

    # Show final status messages
    if [ "$check_only" = "true" ]; then
        if [ $missing -eq 0 ]; then log_and_console "There is nothing to install.";
        elif [ $missing -eq 1 ]; then log_and_console "INFO: 1 dependency is missing.";
        else log_and_console "INFO: $missing dependencies are missing."; fi
    else
        if [ $installed -gt 0 ] && [ $failed -eq 0 ] && [ $missing -eq 0 ]; then
            if [ $installed -eq ${#pkg_list[@]} ]; then
                log_and_console "There is nothing to install."
            else
                log_and_console "All missing dependencies were installed."
            fi
        elif [ $missing -eq 0 ] && [ $failed -eq 0 ]; then
            log_and_console "There is nothing to install."
        elif [ $failed -gt 0 ]; then
            error_exit "Installation failed: $failed packages"
        fi
    fi

    test "$check_only" != "true" && test $failed -gt 0 && error_exit "Installation failed: $failed packages"
    test "${SILENT:-}" || log "Summary: $installed installed, $missing missing, $failed failed"
    test $missing -eq 0
}

show_usage() {
    printf 'ISAC Dependency Installer\n\nUsage: %s [OPTIONS]\n\nOptions:\n  -c, --check    Check for missing dependencies\n  -s, --silent   Silent mode (no console output, only logs)\n  -h, --help     Show help\n\nBehavior:\n  No parameters: Installs any missing dependencies.\n\nNote:\n  If no options are provided, the installer will attempt to install all missing dependencies automatically.\n\nExit: 0=success, 1=missing/error\n' "$0"
}

main() {
    local check_only=""
    SILENT="false"
    init_env "$@"

    while test $# -gt 0; do
        case $1 in
            -c|--check-only) check_only="true" ;;
            -s|--silent) SILENT="true" ;;
            -h|--help) show_usage; exit 0 ;;
            *) error_exit "Unknown option: $1" ;;
        esac; shift
    done

    # Print/log installer started only after SILENT is set and only if not silent
    if [ "$EUID" -eq 0 ]; then
        if [ "$SILENT" != "true" ]; then
            log_and_console "ISAC Dependency Installer started"
        else
            log "ISAC Dependency Installer started"
        fi
    fi

    local platform
    platform=$(detect_platform) || exit 1
    log "Platform detected: $platform"

    if process_packages "$platform" "$check_only"; then
        exit 0
    else
        test "$check_only" = "true" && exit 1 || exit 0
    fi
}

main "$@"
